classdef selfe < dynamicprops
    % obj=selfe(DirIn) - A netcdf style data class to assist reading 
    % SELFE output data from "DirIn", for matlab 2008a (and later
    % versions?).  
    %
    % The constructed object is a wrapper for the "sz_readHeader" and
    % "sz_readTimeStep" scripts provided in m-elio and require data in a binary 
    % Data Format v5.00 (sz-levels) and offers simple alternative to indexing 
    % output files.
    %
    % Index references are strictly in this order:
    % time, elem, vert, u/v
    % e.g. selfeobj.hvel(2,1,1,1) would access u velocity for the 2nd timestep, 1st
    % element and the bottom layer.
    %
    % Note the following important information:
    % hgrid.gr3, vgrid.in and param.in* files must be present in "DirIn", as well
    % output data.  
    %
    % * in "param.in" start date needs to be at the start of the param file, 1st two lines of the 
    % param file may look like this for e.g.
    % ! Note require start time at start of param file for selfe data object...
    % ! 20/10/2010 10:00
    %
    % The directory "DirIn" the must be clean with only combined output files ,
    % (and hgrid.gr3, vgrid.in, param.in) and no missing files (i.e. file 1,2,3,4... must all be present). 
    % Uncombined files should be moved to a seperate directory within or outside of
    %  the "DirIn" directory.
    %
    % Note that filenames and date format ("dd/mm/yyyy HH:MM" format) are hardwired in
    % "/selfeutility/initSELFEvar.m" as:
    %
    %     result.param_file='param.in';
    %     result.grid_file='hgrid.gr3';
    %     result.vgrid_file='vgrid.in';
    %     result.dateFormat='dd/mm/yyyy HH:MM'
    %
    % WARNING: as file access occurs in the background (i.e. calls to "sz_readTimeStep") this object it may 
    % be slow to respond, particularly for a large number of files.
    % Therefore be careful accessing the time dimension with a ":"
    % e.g. selfeobj.hvel(:,:,:,:) may be slow to respond or run out of memory if there are a large 
    % number of files.
    % 
    % Example of use using provided example files, trisurf plot of surface
    % elevations at time 1:
    % %load SELFE object from Example directory provided
    % dirin=which('selfe');
    % [dirin file]=fileparts(dirin);
    % cd(dirin)
    % slf=selfe('..\Example\');
    % %plot elevations
    % figure; 
    % trisurf(slf.elem,slf.x,slf.y,slf.elev(1,:)'); 
    % view([0 90]);
    % shading interp;
    %
    % ---------------------------------------------------------------------
    % 
    %  2 September 2011, version 1 Beta for testing
    %  7 March 2012, version 1.3 still Beta for testing
    %              - fixed an issue with outputs that didn't allow access
    %                to files where the initial files were missing.
    %              - updated description to ensure it is made clear that
    %              object only currently usable with Data Format v5.00
    %              (sz-levels) files.
    %              - fixed bug with single time call to "*.63" files
    %  21 March 2014 version 1.5 still Beta for testing
    %              - fixed bug in file loading, was only loading first file
    %
    %  Ben Knight, Cawthron Institute, 2011
    
    
    properties
        time
        matTime
        x
        y
        relz
        depth
        elem
        datinfo
    end %properties
    
    methods
        %% constructor
        function obj=selfe(DirIn)
            if nargin<1;
                return;
            end
            dat=initSELFE(DirIn); 
            
            %get critical grid information
            obj.time=dat.time(:,1); %time in seconds
            obj.matTime=dat.datetime(:); %matlab time
            obj.x=dat.hgrid.x;  %node x
            obj.y=dat.hgrid.y;  %node y
            obj.depth=dat.hgrid.depth; %node z
            %get relative depths.. not exact! 
            
            obj.relz=sb_computeZlevels(obj.depth, 0*obj.depth, dat.vgrid)./repmat(obj.depth,[1 dat.vgrid.nvrt]);
            obj.elem=dat.hgrid.elem(:,3:5);
            %clear unnessecary data
            rmfield(dat,{'datetime','hgrid'});
            %keep useful data
            obj.datinfo=dat;
            %clear unnessecary data
            clear dat;
            
            %add file specific properties
%             count=1;
            for vars=obj.datinfo.varNames'
                var=char(vars);
                prop=obj.addprop(var);
                eval(['obj.' var '=selfevar(''' var ''');']); %create selfevar object
%                 obj.findprop(var);
%                  eval(['prop.GetMethod = @get_datafromfile;']);
%                 count=count+1;
            end
        end  %constructor
        
        %% display function
        function disp(obj)
            for fld=fields(obj)';
                fld2=char(fld);
                sz=size(eval(['obj.' fld2]));
                szstr='';
                for i=1:length(sz)
                    szstr=[szstr 'x' num2str(sz(i))];
                end
                szstr(1)=[];
                type=class(eval(['obj.' fld2]));
                switch type
                case 'selfevar'
                    disp(eval(['obj.' fld2]));
                otherwise      
                    disp([char(fld) ': [' szstr ' ' type ']'])
                end %switch
            end
        end
        
        %% subsref for "varibles" with associated files
        function value = subsref(obj, S)
            %get data from oject
            theVar = S(1).subs;
            if ~isstr(theVar)
                error('SELFEobj: string variable required')
            end
            if length(S)>1
                eval(['value = subsref(obj.' theVar ', S(2:end));']);
            else
                eval(['value = obj.' theVar ';']);
            end %if length(S)>1
         end
        
        %% access datinfo data
        function value = get.datinfo(obj)
            value=obj.datinfo;
        end
%         
%         function value = get_datafromfile(obj)
% %             S=subsref(obj,substruct('.','blah'));
%             value=1;
%         end
    end
    
    
end %classdef